// Utility Functions

// Format file size
function formatFileSize(bytes) {
    if (bytes === 0) return '0 B';
    const k = 1024;
    const sizes = ['B', 'KB', 'MB', 'GB', 'TB'];
    const i = Math.floor(Math.log(bytes) / Math.log(k));
    return Math.round(bytes / Math.pow(k, i) * 100) / 100 + ' ' + sizes[i];
}

// Format date
function formatDate(dateString) {
    const date = new Date(dateString);
    const now = new Date();
    const diff = now - date;
    const seconds = Math.floor(diff / 1000);
    const minutes = Math.floor(seconds / 60);
    const hours = Math.floor(minutes / 60);
    const days = Math.floor(hours / 24);

    if (days > 7) {
        return date.toLocaleDateString();
    } else if (days > 0) {
        return `${days} day${days > 1 ? 's' : ''} ago`;
    } else if (hours > 0) {
        return `${hours} hour${hours > 1 ? 's' : ''} ago`;
    } else if (minutes > 0) {
        return `${minutes} minute${minutes > 1 ? 's' : ''} ago`;
    } else {
        return 'Just now';
    }
}

// Get file icon based on type/extension
function getFileIcon(fileName, fileType) {
    if (fileType === 'folder') {
        return '<i class="fas fa-folder text-warning"></i>'; // Yellow/Gold for folders
    }

    const ext = fileName.split('.').pop().toLowerCase();
    const iconMap = {
        // Documents
        'pdf': '<i class="fas fa-file-pdf text-danger"></i>',
        'doc': '<i class="fas fa-file-word text-primary"></i>',
        'docx': '<i class="fas fa-file-word text-primary"></i>',
        'txt': '<i class="fas fa-file-alt text-secondary"></i>',
        'rtf': '<i class="fas fa-file-alt text-secondary"></i>',

        // Spreadsheets
        'xls': '<i class="fas fa-file-excel text-success"></i>',
        'xlsx': '<i class="fas fa-file-excel text-success"></i>',
        'csv': '<i class="fas fa-file-csv text-success"></i>',

        // Presentations
        'ppt': '<i class="fas fa-file-powerpoint text-warning"></i>',
        'pptx': '<i class="fas fa-file-powerpoint text-warning"></i>',

        // Images
        'jpg': '<i class="fas fa-file-image text-info"></i>',
        'jpeg': '<i class="fas fa-file-image text-info"></i>',
        'png': '<i class="fas fa-file-image text-info"></i>',
        'gif': '<i class="fas fa-file-image text-info"></i>',
        'svg': '<i class="fas fa-file-image text-info"></i>',
        'webp': '<i class="fas fa-file-image text-info"></i>',

        // Videos
        'mp4': '<i class="fas fa-file-video text-danger"></i>',
        'avi': '<i class="fas fa-file-video text-danger"></i>',
        'mov': '<i class="fas fa-file-video text-danger"></i>',
        'wmv': '<i class="fas fa-file-video text-danger"></i>',
        'webm': '<i class="fas fa-file-video text-danger"></i>',

        // Audio
        'mp3': '<i class="fas fa-file-audio text-warning"></i>',
        'wav': '<i class="fas fa-file-audio text-warning"></i>',
        'ogg': '<i class="fas fa-file-audio text-warning"></i>',
        'flac': '<i class="fas fa-file-audio text-warning"></i>',

        // Archives
        'zip': '<i class="fas fa-file-archive text-warning"></i>',
        'rar': '<i class="fas fa-file-archive text-warning"></i>',
        '7z': '<i class="fas fa-file-archive text-warning"></i>',
        'tar': '<i class="fas fa-file-archive text-warning"></i>',
        'gz': '<i class="fas fa-file-archive text-warning"></i>',

        // Code
        'js': '<i class="fab fa-js text-warning"></i>',
        'html': '<i class="fab fa-html5 text-danger"></i>',
        'css': '<i class="fab fa-css3-alt text-primary"></i>',
        'json': '<i class="fas fa-code text-secondary"></i>',
        'xml': '<i class="fas fa-code text-secondary"></i>',
        'py': '<i class="fab fa-python text-primary"></i>',
        'java': '<i class="fab fa-java text-danger"></i>',
        'cpp': '<i class="fas fa-code text-primary"></i>',
        'c': '<i class="fas fa-code text-primary"></i>',
        'php': '<i class="fab fa-php text-primary"></i>',
        'rb': '<i class="fas fa-gem text-danger"></i>',
        'go': '<i class="fas fa-code text-info"></i>',
        'rs': '<i class="fas fa-code text-dark"></i>',
    };

    return iconMap[ext] || '<i class="fas fa-file text-secondary"></i>';
}

// Show toast notification
function showToast(message, type = 'info') {
    const container = document.getElementById('toast-container') || createToastContainer();

    const toast = document.createElement('div');
    toast.className = `toast ${type}`;

    let icon = 'info-circle';
    if (type === 'success') icon = 'check-circle';
    if (type === 'error') icon = 'exclamation-circle';
    if (type === 'warning') icon = 'exclamation-triangle';

    toast.innerHTML = `
    <div style="display: flex; align-items: start; gap: 12px;">
      <div style="flex-shrink: 0; font-size: 20px;">
        <i class="fas fa-${icon}"></i>
      </div>
      <div style="flex: 1;">
        <div style="font-weight: 500; margin-bottom: 4px;">
          ${type.charAt(0).toUpperCase() + type.slice(1)}
        </div>
        <div style="font-size: 14px; color: var(--text-secondary);">
          ${message}
        </div>
      </div>
      <button onclick="this.parentElement.parentElement.remove()" style="flex-shrink: 0; color: var(--text-secondary); padding: 4px; background: none; border: none; cursor: pointer;">
        <i class="fas fa-times"></i>
      </button>
    </div>
  `;

    container.appendChild(toast);

    // Auto remove after 5 seconds
    setTimeout(() => {
        toast.style.animation = 'fadeOut 0.3s ease-out';
        setTimeout(() => toast.remove(), 300);
    }, 5000);
}

function createToastContainer() {
    const container = document.createElement('div');
    container.id = 'toast-container';
    container.className = 'toast-container';
    document.body.appendChild(container);
    return container;
}

// Show loading overlay
function showLoading() {
    const overlay = document.createElement('div');
    overlay.id = 'loading-overlay';
    overlay.className = 'loading-overlay';
    overlay.innerHTML = '<div class="spinner"></div>';
    document.body.appendChild(overlay);
}

function hideLoading() {
    const overlay = document.getElementById('loading-overlay');
    if (overlay) {
        overlay.remove();
    }
}



// Debounce function
function debounce(func, wait) {
    let timeout;
    return function executedFunction(...args) {
        const later = () => {
            clearTimeout(timeout);
            func(...args);
        };
        clearTimeout(timeout);
        timeout = setTimeout(later, wait);
    };
}

// Escape HTML
function escapeHtml(text) {
    const div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML;
}

// Parse path
function parsePath(path) {
    const parts = path.split('/').filter(p => p);
    return parts;
}

// Join path
function joinPath(...parts) {
    return '/' + parts.filter(p => p).join('/');
}

// Get parent path
function getParentPath(path) {
    const parts = parsePath(path);
    parts.pop();
    return '/' + parts.join('/');
}

// Storage helpers
const storage = {
    get(key) {
        try {
            const item = localStorage.getItem(key);
            return item ? JSON.parse(item) : null;
        } catch {
            return null;
        }
    },

    set(key, value) {
        try {
            localStorage.setItem(key, JSON.stringify(value));
        } catch (error) {
            console.error('Storage error:', error);
        }
    },

    remove(key) {
        localStorage.removeItem(key);
    },

    clear() {
        localStorage.clear();
    }
};

// Check if user is authenticated
function isAuthenticated() {
    return storage.get('user') !== null;
}

// Get current user
function getCurrentUser() {
    return storage.get('user');
}

// Set current user
function setCurrentUser(user) {
    storage.set('user', user);
}

// Clear current user
function clearCurrentUser() {
    storage.remove('user');
}

// Redirect to login
function redirectToLogin() {
    window.location.href = '/pages/login.html';
}

// Redirect to dashboard
function redirectToDashboard() {
    window.location.href = '/pages/dashboard.html';
}
