// API Configuration
const isLocal = ['localhost', '127.0.0.1'].includes(window.location.hostname);
const API_BASE_URL = isLocal ? `http://${window.location.hostname}:3000/api` : '/api';

// API Client
class APIClient {
    constructor() {
        this.baseURL = API_BASE_URL;
    }

    async request(endpoint, options = {}) {
        let url = `${this.baseURL}${endpoint}`;

        // Add cache buster for GET requests
        if (!options.method || options.method === 'GET') {
            const separator = url.includes('?') ? '&' : '?';
            url += `${separator}_t=${Date.now()}`;
        }

        const config = {
            ...options,
            headers: {
                'Content-Type': 'application/json',
                ...options.headers,
            },
            credentials: 'include', // Include cookies
        };

        // Remove Content-Type for FormData
        if (options.body instanceof FormData) {
            delete config.headers['Content-Type'];
        }

        try {
            const response = await fetch(url, config);
            const data = await response.json();

            if (!response.ok) {
                const errorMessage = data.details
                    ? `${data.error}: ${data.details}`
                    : (data.error || 'Request failed');
                throw new Error(errorMessage);
            }

            return data;
        } catch (error) {
            console.error('API Error:', error);
            throw error;
        }
    }

    // Auth endpoints
    async login(username, password) {
        return this.request('/auth/login', {
            method: 'POST',
            body: JSON.stringify({ username, password }),
        });
    }

    async logout() {
        return this.request('/auth/logout', {
            method: 'POST',
        });
    }

    async getCurrentUser() {
        return this.request('/auth/me');
    }

    async register(username, password, role, permissions) {
        return this.request('/auth/register', {
            method: 'POST',
            body: JSON.stringify({ username, password, role, permissions }),
        });
    }

    // File endpoints
    async listFiles(path = '/') {
        return this.request(`/files?path=${encodeURIComponent(path)}`);
    }

    async uploadFile(file, path = '/') {
        const formData = new FormData();
        formData.append('file', file);
        formData.append('path', path);

        return this.request('/files/upload', {
            method: 'POST',
            body: formData,
        });
    }

    async downloadFile(fileId) {
        const url = `${this.baseURL}/files/download/${fileId}`;
        window.open(url, '_blank');
    }

    async renameFile(fileId, newName) {
        return this.request(`/files/rename/${fileId}`, {
            method: 'PUT',
            body: JSON.stringify({ newName }),
        });
    }

    async moveFile(fileId, newPath) {
        return this.request(`/files/move/${fileId}`, {
            method: 'PUT',
            body: JSON.stringify({ newPath }),
        });
    }

    async copyFile(fileId, destPath) {
        return this.request(`/files/copy/${fileId}`, {
            method: 'POST',
            body: JSON.stringify({ destPath }),
        });
    }

    async deleteFile(fileId) {
        return this.request(`/files/${fileId}`, {
            method: 'DELETE',
        });
    }

    async listTrash() {
        return this.request('/files/trash/list');
    }

    async restoreFile(fileId) {
        return this.request(`/files/restore/${fileId}`, {
            method: 'PUT'
        });
    }

    async emptyTrash() {
        return this.request('/files/trash/empty', {
            method: 'DELETE'
        });
    }

    async compressFiles(fileIds, archiveName, currentPath) {
        return this.request('/files/compress', {
            method: 'POST',
            body: JSON.stringify({ fileIds, archiveName, currentPath })
        });
    }

    async extractFile(fileId, extractPath) {
        return this.request('/files/extract', {
            method: 'POST',
            body: JSON.stringify({ fileId, extractPath })
        });
    }

    async createFolder(name, path = '/') {
        return this.request('/files/folder', {
            method: 'POST',
            body: JSON.stringify({ name, path }),
        });
    }

    async searchFiles(query) {
        return this.request(`/files/search?q=${encodeURIComponent(query)}`);
    }

    async getStorageStats() {
        return this.request('/files/stats');
    }

    async getFileInfo(fileId) {
        return this.request(`/files/info/${fileId}`);
    }

    // Terminal endpoints
    async getFileContent(fileId) {
        return this.request(`/files/content/${fileId}`);
    }

    async updateFileContent(fileId, content) {
        return this.request(`/files/content/${fileId}`, {
            method: 'PUT',
            body: JSON.stringify({ content })
        });
    }

    async executeCommand(command) {
        return this.request('/terminal/execute', {
            method: 'POST',
            body: JSON.stringify({ command }),
        });
    }

    async getCurrentDirectory() {
        return this.request('/terminal/pwd');
    }

    // User management endpoints
    async getUsers() {
        return this.request('/users');
    }

    async createUser(username, password, role, permissions) {
        return this.request('/users', {
            method: 'POST',
            body: JSON.stringify({ username, password, role, permissions }),
        });
    }

    async updatePermissions(userId, permissions) {
        return this.request(`/users/${userId}/permissions`, {
            method: 'PUT',
            body: JSON.stringify({ permissions }),
        });
    }

    async deleteUser(userId) {
        return this.request(`/users/${userId}`, {
            method: 'DELETE',
        });
    }

    async getUser(userId) {
        return this.request(`/users/${userId}`);
    }

    // Logs endpoints
    async getLogs(filters = {}) {
        const params = new URLSearchParams(filters);
        return this.request(`/logs?${params}`);
    }

    async getUserLogs(userId, limit = 50) {
        return this.request(`/logs/user/${userId}?limit=${limit}`);
    }

    async getRecentLogs(limit = 50) {
        return this.request(`/logs/recent?limit=${limit}`);
    }
}

// Export singleton instance
const api = new APIClient();
