const { supabase } = require('../config/database');

class LogService {
    /**
     * Create an activity log entry
     */
    async createLog(userId, action, target = null, details = {}, ipAddress = null) {
        try {
            const { data, error } = await supabase
                .from('activity_logs')
                .insert({
                    user_id: userId,
                    action,
                    target,
                    details,
                    ip_address: ipAddress
                })
                .select()
                .single();

            if (error) {
                console.error('Error creating log:', error);
                return null;
            }

            return data;
        } catch (error) {
            console.error('Error in createLog:', error);
            return null;
        }
    }

    /**
     * Get logs with optional filters
     */
    async getLogs(filters = {}, limit = 100, offset = 0) {
        try {
            let query = supabase
                .from('activity_logs')
                .select(`
          *,
          users:user_id (username, role)
        `)
                .order('timestamp', { ascending: false })
                .range(offset, offset + limit - 1);

            // Apply filters
            if (filters.userId) {
                query = query.eq('user_id', filters.userId);
            }

            if (filters.action) {
                query = query.eq('action', filters.action);
            }

            if (filters.startDate) {
                query = query.gte('timestamp', filters.startDate);
            }

            if (filters.endDate) {
                query = query.lte('timestamp', filters.endDate);
            }

            const { data, error } = await query;

            if (error) {
                console.error('Error fetching logs:', error);
                return [];
            }

            return data;
        } catch (error) {
            console.error('Error in getLogs:', error);
            return [];
        }
    }

    /**
     * Get logs for a specific user
     */
    async getUserLogs(userId, limit = 50) {
        return this.getLogs({ userId }, limit);
    }

    /**
     * Get recent logs
     */
    async getRecentLogs(limit = 50) {
        return this.getLogs({}, limit);
    }

    /**
     * Delete old logs (cleanup)
     */
    async deleteOldLogs(daysToKeep = 90) {
        try {
            const cutoffDate = new Date();
            cutoffDate.setDate(cutoffDate.getDate() - daysToKeep);

            const { error } = await supabase
                .from('activity_logs')
                .delete()
                .lt('timestamp', cutoffDate.toISOString());

            if (error) {
                console.error('Error deleting old logs:', error);
                return false;
            }

            return true;
        } catch (error) {
            console.error('Error in deleteOldLogs:', error);
            return false;
        }
    }
}

module.exports = new LogService();
