const express = require('express');
const router = express.Router();
const logService = require('../services/logService');
const { authenticate } = require('../middleware/auth');
const { requireAdmin } = require('../middleware/permissions');

/**
 * GET /api/logs
 * Get activity logs (admin only)
 */
router.get('/', authenticate, requireAdmin, async (req, res) => {
    try {
        const { limit = 50, offset = 0, userId, action, startDate, endDate } = req.query;

        const filters = {};
        if (userId) filters.userId = userId;
        if (action) filters.action = action;
        if (startDate) filters.startDate = startDate;
        if (endDate) filters.endDate = endDate;

        const logs = await logService.getLogs(filters, parseInt(limit), parseInt(offset));
        res.json({ success: true, logs });
    } catch (error) {
        console.error('Get logs error:', error);
        res.status(500).json({ error: 'Failed to get logs' });
    }
});

/**
 * GET /api/logs/user/:userId
 * Get logs for a specific user (admin only)
 */
router.get('/user/:userId', authenticate, requireAdmin, async (req, res) => {
    try {
        const { userId } = req.params;
        const { limit = 50 } = req.query;

        const logs = await logService.getUserLogs(userId, parseInt(limit));
        res.json({ success: true, logs });
    } catch (error) {
        console.error('Get user logs error:', error);
        res.status(500).json({ error: 'Failed to get user logs' });
    }
});

/**
 * GET /api/logs/recent
 * Get recent logs (admin only)
 */
router.get('/recent', authenticate, requireAdmin, async (req, res) => {
    try {
        const { limit = 50 } = req.query;
        const logs = await logService.getRecentLogs(parseInt(limit));
        res.json({ success: true, logs });
    } catch (error) {
        console.error('Get recent logs error:', error);
        res.status(500).json({ error: 'Failed to get recent logs' });
    }
});

module.exports = router;
