const express = require('express');
const router = express.Router();
const authService = require('../services/authService');
const { authenticate } = require('../middleware/auth');
const { requireAdmin } = require('../middleware/permissions');
const { loginLimiter } = require('../middleware/rateLimiter');

/**
 * POST /api/auth/login
 * User login
 */
router.post('/login', loginLimiter, async (req, res) => {
    try {
        const { username, password } = req.body;

        if (!username || !password) {
            return res.status(400).json({ error: 'Username and password are required' });
        }

        const ipAddress = req.ip || req.connection.remoteAddress;
        const result = await authService.login(username, password, ipAddress);

        // Set session cookie
        res.cookie('session_token', result.session.token, {
            httpOnly: true,
            secure: process.env.NODE_ENV === 'production',
            sameSite: 'strict',
            maxAge: 24 * 60 * 60 * 1000 // 24 hours
        });

        res.json({
            success: true,
            user: result.user,
            session: result.session
        });
    } catch (error) {
        console.error('Login error:', error);
        res.status(401).json({ error: error.message || 'Login failed' });
    }
});

/**
 * POST /api/auth/logout
 * User logout
 */
router.post('/logout', authenticate, async (req, res) => {
    try {
        const ipAddress = req.ip || req.connection.remoteAddress;
        await authService.logout(
            req.sessionToken,
            req.user.id,
            req.user.username,
            req.user.role,
            ipAddress
        );

        // Clear session cookie
        res.clearCookie('session_token');

        res.json({ success: true, message: 'Logged out successfully' });
    } catch (error) {
        console.error('Logout error:', error);
        res.status(500).json({ error: 'Logout failed' });
    }
});

/**
 * GET /api/auth/me
 * Get current user info
 */
router.get('/me', authenticate, async (req, res) => {
    try {
        const user = await authService.getUserById(req.user.id);
        res.json({ success: true, user });
    } catch (error) {
        console.error('Get user error:', error);
        res.status(500).json({ error: 'Failed to get user info' });
    }
});

/**
 * POST /api/auth/register
 * Create new user (admin only)
 */
router.post('/register', authenticate, requireAdmin, async (req, res) => {
    try {
        const { username, password, role, permissions } = req.body;

        if (!username || !password) {
            return res.status(400).json({ error: 'Username and password are required' });
        }

        if (username.length < 3) {
            return res.status(400).json({ error: 'Username must be at least 3 characters' });
        }

        if (password.length < 6) {
            return res.status(400).json({ error: 'Password must be at least 6 characters' });
        }

        const user = await authService.createUser(username, password, role || 'staff', permissions);

        // Log user creation
        const discordService = require('../services/discordService');
        await discordService.logUserCreate(req.user.username, username, user.role);

        const { password_hash, ...userWithoutPassword } = user;
        res.status(201).json({ success: true, user: userWithoutPassword });
    } catch (error) {
        console.error('Register error:', error);
        res.status(400).json({ error: error.message || 'Registration failed' });
    }
});

module.exports = router;
