const path = require('path');
require('dotenv').config({ path: path.join(__dirname, '.env') });
const fileService = require('./src/services/fileService');
const { supabase } = require('./src/config/database');

async function run() {
    console.log('Starting debug script...');
    try {
        console.log('Fetching a valid user...');
        const { data: users, error: userError } = await supabase.from('users').select('id, username').limit(1);
        if (userError) {
            console.error('Error fetching users:', userError);
            return;
        }
        if (!users || !users.length) {
            console.error('No users found. Cannot proceed.');
            return;
        }
        const user = users[0];
        console.log('Using user:', user);

        console.log('Listing files...');
        const files = await fileService.listFiles('/', user.id);
        console.log(`Found ${files.length} files`);

        let targetFile = files.find(f => f.type === 'file');

        // Use existing file if available to test real download
        if (!targetFile) {
            console.log('No existing file found, creating a test file...');
            targetFile = await fileService.uploadFile({
                originalname: `test-compress-${Date.now()}.txt`,
                buffer: Buffer.from('Hello world test compression'),
                mimetype: 'text/plain',
                size: 28
            }, '/', user.id, user.username);
            console.log('Created test file:', targetFile);
        } else {
            console.log('Using existing file:', targetFile.name, targetFile.id);
        }

        console.log('Attempting to compress...');
        await fileService.compressFiles([targetFile.id], 'debug-archive', '/', user.id, user.username);
        console.log('Compression successful!');

    } catch (error) {
        console.error('Debug script failed:');
        console.error(error);
        if (error.stack) console.error(error.stack);
    } finally {
        process.exit(0);
    }
}

run();
