
const axios = require('axios');
const {
    DISCORD_WEBHOOK_URL,
    DISCORD_WEBHOOK_USER_AUTH,
    DISCORD_WEBHOOK_USER_CREATE,
    DISCORD_WEBHOOK_FILE_UPLOAD,
    DISCORD_WEBHOOK_FOLDER_CREATE,
    DISCORD_WEBHOOK_FILE_RENAME,
    DISCORD_WEBHOOK_FILE_MOVE,
    DISCORD_WEBHOOK_FILE_DELETE,
    enabled
} = require('../config/discord');

class DiscordService {
    /**
     * Send a formatted embed to Discord webhook
     */
    async sendEmbed(title, fields, color = 5763719, webhookUrl = DISCORD_WEBHOOK_URL) {
        if (!enabled && !webhookUrl) {
            console.log('Discord webhook disabled or missing, skipping notification');
            return;
        }

        // Use specific webhook if provided, otherwise default. If specific missing, fallback or skip?
        // Logic: passed webhookUrl takes precedence. If it's undefined (e.g. auth webhook not set), fallback to default?
        // Or just skip if specific one is missing. 
        // User explicitly asked for separate ones. If separated one is missing, maybe fallback to default is safer.
        const targetUrl = webhookUrl || DISCORD_WEBHOOK_URL;

        if (!targetUrl) return;

        try {
            const embed = {
                embeds: [{
                    title,
                    color,
                    fields,
                    timestamp: new Date().toISOString()
                }]
            };

            await axios.post(targetUrl, embed);
        } catch (error) {
            console.error('Error sending Discord webhook:', error.message);
        }
    }

    /**
     * Log user login
     */
    async logLogin(username, role) {
        await this.sendEmbed(
            '🔐 User Login',
            [
                { name: 'Username', value: username, inline: true },
                { name: 'Role', value: role, inline: true },
                { name: 'Time', value: new Date().toLocaleString(), inline: false }
            ],
            5763719, // Blue
            DISCORD_WEBHOOK_USER_AUTH
        );
    }

    /**
     * Log user logout
     */
    async logLogout(username, role) {
        await this.sendEmbed(
            '🚪 User Logout',
            [
                { name: 'Username', value: username, inline: true },
                { name: 'Role', value: role, inline: true },
                { name: 'Time', value: new Date().toLocaleString(), inline: false }
            ],
            10070709, // Gray
            DISCORD_WEBHOOK_USER_AUTH
        );
    }

    /**
     * Log file upload
     */
    async logFileUpload(username, fileName, fileSize, filePath) {
        const sizeInMB = (fileSize / (1024 * 1024)).toFixed(2);
        await this.sendEmbed(
            '📤 File Uploaded',
            [
                { name: 'User', value: username, inline: true },
                { name: 'File', value: fileName, inline: true },
                { name: 'Size', value: `${sizeInMB} MB`, inline: true },
                { name: 'Path', value: filePath || '/', inline: false }
            ],
            3066993, // Green
            DISCORD_WEBHOOK_FILE_UPLOAD
        );
    }

    /**
     * Log file deletion
     */
    async logFileDelete(username, fileName, filePath) {
        await this.sendEmbed(
            '🗑️ File Deleted',
            [
                { name: 'User', value: username, inline: true },
                { name: 'File', value: fileName, inline: true },
                { name: 'Path', value: filePath || '/', inline: false }
            ],
            15158332, // Red
            DISCORD_WEBHOOK_FILE_DELETE
        );
    }

    /**
     * Log file rename
     */
    async logFileRename(username, oldName, newName, filePath) {
        await this.sendEmbed(
            '✏️ File Renamed',
            [
                { name: 'User', value: username, inline: true },
                { name: 'Old Name', value: oldName, inline: true },
                { name: 'New Name', value: newName, inline: true },
                { name: 'Path', value: filePath || '/', inline: false }
            ],
            15844367, // Gold
            DISCORD_WEBHOOK_FILE_RENAME
        );
    }

    /**
     * Log file move
     */
    async logFileMove(username, fileName, fromPath, toPath) {
        await this.sendEmbed(
            '📦 File Moved',
            [
                { name: 'User', value: username, inline: true },
                { name: 'File', value: fileName, inline: true },
                { name: 'From', value: fromPath || '/', inline: true },
                { name: 'To', value: toPath || '/', inline: true }
            ],
            3447003, // Blue
            DISCORD_WEBHOOK_FILE_MOVE
        );
    }

    /**
     * Log file edit
     */
    async logFileEdit(username, fileName, filePath) {
        await this.sendEmbed(
            '📝 File Edited',
            [
                { name: 'User', value: username, inline: true },
                { name: 'File', value: fileName, inline: true },
                { name: 'Path', value: filePath || '/', inline: false }
            ],
            16776960, // Yellow
            process.env.DISCORD_EDIT_WEBHOOK_URL // Use env directly or config if updated
        );
    }

    /**
     * Log folder creation
     */
    async logFolderCreate(username, folderName, folderPath) {
        await this.sendEmbed(
            '📁 Folder Created',
            [
                { name: 'User', value: username, inline: true },
                { name: 'Folder', value: folderName, inline: true },
                { name: 'Path', value: folderPath || '/', inline: false }
            ],
            3066993, // Green
            DISCORD_WEBHOOK_FOLDER_CREATE
        );
    }

    /**
     * Log permission change
     */
    async logPermissionChange(adminUsername, targetUsername, permissions) {
        const permList = Object.entries(permissions)
            .map(([key, value]) => `${key}: ${value ? '✅' : '❌'} `)
            .join('\n');

        await this.sendEmbed(
            '🔑 Permissions Updated',
            [
                { name: 'Admin', value: adminUsername, inline: true },
                { name: 'Target User', value: targetUsername, inline: true },
                { name: 'Permissions', value: permList, inline: false }
            ],
            15844367, // Gold
            DISCORD_WEBHOOK_URL // Default
        );
    }

    /**
     * Log user creation
     */
    async logUserCreate(adminUsername, newUsername, role) {
        await this.sendEmbed(
            '👤 User Created',
            [
                { name: 'Admin', value: adminUsername, inline: true },
                { name: 'New User', value: newUsername, inline: true },
                { name: 'Role', value: role, inline: true }
            ],
            3066993, // Green
            DISCORD_WEBHOOK_USER_CREATE
        );
    }
}

module.exports = new DiscordService();
