const bcrypt = require('bcrypt');
const { v4: uuidv4 } = require('uuid');
const { supabase } = require('../config/database');
const logService = require('./logService');
const discordService = require('./discordService');

const SALT_ROUNDS = 12;
const SESSION_EXPIRY_HOURS = parseInt(process.env.SESSION_EXPIRY_HOURS) || 24;

class AuthService {
    /**
     * Hash a password
     */
    async hashPassword(password) {
        return bcrypt.hash(password, SALT_ROUNDS);
    }

    /**
     * Compare password with hash
     */
    async comparePassword(password, hash) {
        return bcrypt.compare(password, hash);
    }

    /**
     * Create a new user (admin only)
     */
    async createUser(username, password, role = 'staff', permissions = null) {
        try {
            // Validate role
            if (!['admin', 'staff', 'public'].includes(role)) {
                throw new Error('Invalid role');
            }

            // Default permissions based on role
            const defaultPermissions = {
                admin: { read: true, write: true, edit: true, move: true, delete: true, restore: true, empty_trash: true },
                staff: { read: true, write: false, edit: true, move: false, delete: false, restore: true, empty_trash: false },
                public: { read: true, write: false, edit: false, move: false, delete: false, restore: false, empty_trash: false }
            };

            const userPermissions = permissions || defaultPermissions[role];

            // Hash password
            const passwordHash = await this.hashPassword(password);

            // Create user
            const { data, error } = await supabase
                .from('users')
                .insert({
                    username,
                    password_hash: passwordHash,
                    role,
                    permissions: userPermissions
                })
                .select()
                .single();

            if (error) {
                if (error.code === '23505') { // Unique constraint violation
                    throw new Error('Username already exists');
                }
                throw error;
            }

            return data;
        } catch (error) {
            console.error('Error creating user:', error);
            throw error;
        }
    }

    /**
     * Login user
     */
    async login(username, password, ipAddress = null) {
        try {
            // Get user
            const { data: user, error } = await supabase
                .from('users')
                .select('*')
                .eq('username', username)
                .single();

            if (error || !user) {
                throw new Error('Invalid credentials');
            }

            // Compare password
            const isValid = await this.comparePassword(password, user.password_hash);
            if (!isValid) {
                throw new Error('Invalid credentials');
            }

            // Create session
            const session = await this.createSession(user.id);

            // Update last login
            await supabase
                .from('users')
                .update({ last_login: new Date().toISOString() })
                .eq('id', user.id);

            // Log activity
            await logService.createLog(user.id, 'login', null, {}, ipAddress);
            await discordService.logLogin(user.username, user.role);

            // Return user without password hash
            const { password_hash, ...userWithoutPassword } = user;
            return {
                user: userWithoutPassword,
                session
            };
        } catch (error) {
            console.error('Error during login:', error);
            throw error;
        }
    }

    /**
     * Logout user
     */
    async logout(sessionToken, userId, username, role, ipAddress = null) {
        try {
            // Delete session
            await supabase
                .from('sessions')
                .delete()
                .eq('token', sessionToken);

            // Log activity
            await logService.createLog(userId, 'logout', null, {}, ipAddress);
            await discordService.logLogout(username, role);

            return true;
        } catch (error) {
            console.error('Error during logout:', error);
            throw error;
        }
    }

    /**
     * Create a session
     */
    async createSession(userId) {
        try {
            const token = uuidv4();
            const expiresAt = new Date();
            expiresAt.setHours(expiresAt.getHours() + SESSION_EXPIRY_HOURS);

            const { data, error } = await supabase
                .from('sessions')
                .insert({
                    user_id: userId,
                    token,
                    expires_at: expiresAt.toISOString()
                })
                .select()
                .single();

            if (error) {
                throw error;
            }

            return data;
        } catch (error) {
            console.error('Error creating session:', error);
            throw error;
        }
    }

    /**
     * Validate session token
     */
    async validateSession(token) {
        try {
            const { data: session, error } = await supabase
                .from('sessions')
                .select(`
          *,
          users (*)
        `)
                .eq('token', token)
                .gt('expires_at', new Date().toISOString())
                .single();

            if (error || !session) {
                return null;
            }

            return session;
        } catch (error) {
            console.error('Error validating session:', error);
            return null;
        }
    }

    /**
     * Clean expired sessions
     */
    async cleanExpiredSessions() {
        try {
            await supabase
                .from('sessions')
                .delete()
                .lt('expires_at', new Date().toISOString());
        } catch (error) {
            console.error('Error cleaning expired sessions:', error);
        }
    }

    /**
     * Get user by ID
     */
    async getUserById(userId) {
        try {
            const { data, error } = await supabase
                .from('users')
                .select('id, username, role, permissions, created_at, last_login')
                .eq('id', userId)
                .single();

            if (error) {
                throw error;
            }

            return data;
        } catch (error) {
            console.error('Error getting user:', error);
            return null;
        }
    }

    /**
     * Get all users (admin only)
     */
    async getAllUsers() {
        try {
            const { data, error } = await supabase
                .from('users')
                .select('id, username, role, permissions, created_at, last_login')
                .order('created_at', { ascending: false });

            if (error) {
                throw error;
            }

            return data;
        } catch (error) {
            console.error('Error getting users:', error);
            return [];
        }
    }

    /**
     * Update user permissions (admin only)
     */
    async updatePermissions(userId, permissions) {
        try {
            const { data, error } = await supabase
                .from('users')
                .update({ permissions })
                .eq('id', userId)
                .select()
                .single();

            if (error) {
                throw error;
            }

            return data;
        } catch (error) {
            console.error('Error updating permissions:', error);
            throw error;
        }
    }

    /**
     * Delete user (admin only)
     */
    async deleteUser(userId) {
        try {
            const { error } = await supabase
                .from('users')
                .delete()
                .eq('id', userId);

            if (error) {
                throw error;
            }

            return true;
        } catch (error) {
            console.error('Error deleting user:', error);
            throw error;
        }
    }
}

module.exports = new AuthService();
