require('dotenv').config();
const { supabase } = require('../config/database');
const fs = require('fs');
const path = require('path');

async function initDatabase() {
    try {
        console.log('Initializing database...\n');

        // Read schema file
        const schemaPath = path.join(__dirname, '../models/schema.sql');
        const schema = fs.readFileSync(schemaPath, 'utf8');

        // Split schema into individual statements
        const statements = schema
            .split(';')
            .map(s => s.trim())
            .filter(s => s.length > 0);

        console.log(`Executing ${statements.length} SQL statements...\n`);

        // Execute each statement
        for (let i = 0; i < statements.length; i++) {
            const statement = statements[i];

            // Skip comments
            if (statement.startsWith('--')) continue;

            try {
                const { error } = await supabase.rpc('exec_sql', { sql: statement });

                if (error) {
                    // Try direct query if RPC fails
                    const { error: queryError } = await supabase.from('_').select('*').limit(0);

                    // For now, just log and continue (Supabase might not support all SQL via API)
                    console.log(`⚠️  Statement ${i + 1}: ${error.message || 'Skipped'}`);
                } else {
                    console.log(`✅ Statement ${i + 1}: Executed successfully`);
                }
            } catch (err) {
                console.log(`⚠️  Statement ${i + 1}: ${err.message}`);
            }
        }

        console.log('\n━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━');
        console.log('Database initialization complete!');
        console.log('━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n');
        console.log('⚠️  NOTE: Some statements may need to be run manually in Supabase SQL Editor:');
        console.log('   1. Go to your Supabase project dashboard');
        console.log('   2. Navigate to SQL Editor');
        console.log('   3. Copy and paste the contents of src/models/schema.sql');
        console.log('   4. Run the SQL script\n');

        process.exit(0);
    } catch (error) {
        console.error('Error initializing database:', error);
        process.exit(1);
    }
}

initDatabase();
