"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.VectorIndexScope = exports.VectorBucketScope = exports.StorageVectorsClient = void 0;
const tslib_1 = require("tslib");
const VectorIndexApi_1 = tslib_1.__importDefault(require("./VectorIndexApi"));
const VectorDataApi_1 = tslib_1.__importDefault(require("./VectorDataApi"));
const VectorBucketApi_1 = tslib_1.__importDefault(require("./VectorBucketApi"));
/**
 *
 * @alpha
 *
 * Main client for interacting with S3 Vectors API
 * Provides access to bucket, index, and vector data operations
 *
 * **Public alpha:** This API is part of a public alpha release and may not be available to your account type.
 *
 * **Usage Patterns:**
 *
 * ```typescript
 * const { data, error } = await supabase
 *  .storage
 *  .vectors
 *  .createBucket('embeddings-prod')
 *
 * // Access index operations via buckets
 * const bucket = supabase.storage.vectors.from('embeddings-prod')
 * await bucket.createIndex({
 *   indexName: 'documents',
 *   dataType: 'float32',
 *   dimension: 1536,
 *   distanceMetric: 'cosine'
 * })
 *
 * // Access vector operations via index
 * const index = bucket.index('documents')
 * await index.putVectors({
 *   vectors: [
 *     { key: 'doc-1', data: { float32: [...] }, metadata: { title: 'Intro' } }
 *   ]
 * })
 *
 * // Query similar vectors
 * const { data } = await index.queryVectors({
 *   queryVector: { float32: [...] },
 *   topK: 5,
 *   returnDistance: true
 * })
 * ```
 */
class StorageVectorsClient extends VectorBucketApi_1.default {
    /**
     * @alpha
     *
     * Creates a StorageVectorsClient that can manage buckets, indexes, and vectors.
     *
     * **Public alpha:** This API is part of a public alpha release and may not be available to your account type.
     *
     * @category Vector Buckets
     * @param url - Base URL of the Storage Vectors REST API.
     * @param options.headers - Optional headers (for example `Authorization`) applied to every request.
     * @param options.fetch - Optional custom `fetch` implementation for non-browser runtimes.
     *
     * @example
     * ```typescript
     * const client = new StorageVectorsClient(url, options)
     * ```
     */
    constructor(url, options = {}) {
        super(url, options.headers || {}, options.fetch);
    }
    /**
     *
     * @alpha
     *
     * Access operations for a specific vector bucket
     * Returns a scoped client for index and vector operations within the bucket
     *
     * **Public alpha:** This API is part of a public alpha release and may not be available to your account type.
     *
     * @category Vector Buckets
     * @param vectorBucketName - Name of the vector bucket
     * @returns Bucket-scoped client with index and vector operations
     *
     * @example
     * ```typescript
     * const bucket = supabase.storage.vectors.from('embeddings-prod')
     * ```
     */
    from(vectorBucketName) {
        return new VectorBucketScope(this.url, this.headers, vectorBucketName, this.fetch);
    }
    /**
     *
     * @alpha
     *
     * Creates a new vector bucket
     * Vector buckets are containers for vector indexes and their data
     *
     * **Public alpha:** This API is part of a public alpha release and may not be available to your account type.
     *
     * @category Vector Buckets
     * @param vectorBucketName - Unique name for the vector bucket
     * @returns Promise with empty response on success or error
     *
     * @example
     * ```typescript
     * const { data, error } = await supabase
     *   .storage
     *   .vectors
     *   .createBucket('embeddings-prod')
     * ```
     */
    createBucket(vectorBucketName) {
        const _super = Object.create(null, {
            createBucket: { get: () => super.createBucket }
        });
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            return _super.createBucket.call(this, vectorBucketName);
        });
    }
    /**
     *
     * @alpha
     *
     * Retrieves metadata for a specific vector bucket
     *
     * **Public alpha:** This API is part of a public alpha release and may not be available to your account type.
     *
     * @category Vector Buckets
     * @param vectorBucketName - Name of the vector bucket
     * @returns Promise with bucket metadata or error
     *
     * @example
     * ```typescript
     * const { data, error } = await supabase
     *   .storage
     *   .vectors
     *   .getBucket('embeddings-prod')
     *
     * console.log('Bucket created:', data?.vectorBucket.creationTime)
     * ```
     */
    getBucket(vectorBucketName) {
        const _super = Object.create(null, {
            getBucket: { get: () => super.getBucket }
        });
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            return _super.getBucket.call(this, vectorBucketName);
        });
    }
    /**
     *
     * @alpha
     *
     * Lists all vector buckets with optional filtering and pagination
     *
     * **Public alpha:** This API is part of a public alpha release and may not be available to your account type.
     *
     * @category Vector Buckets
     * @param options - Optional filters (prefix, maxResults, nextToken)
     * @returns Promise with list of buckets or error
     *
     * @example
     * ```typescript
     * const { data, error } = await supabase
     *   .storage
     *   .vectors
     *   .listBuckets({ prefix: 'embeddings-' })
     *
     * data?.vectorBuckets.forEach(bucket => {
     *   console.log(bucket.vectorBucketName)
     * })
     * ```
     */
    listBuckets() {
        const _super = Object.create(null, {
            listBuckets: { get: () => super.listBuckets }
        });
        return tslib_1.__awaiter(this, arguments, void 0, function* (options = {}) {
            return _super.listBuckets.call(this, options);
        });
    }
    /**
     *
     * @alpha
     *
     * Deletes a vector bucket (bucket must be empty)
     * All indexes must be deleted before deleting the bucket
     *
     * **Public alpha:** This API is part of a public alpha release and may not be available to your account type.
     *
     * @category Vector Buckets
     * @param vectorBucketName - Name of the vector bucket to delete
     * @returns Promise with empty response on success or error
     *
     * @example
     * ```typescript
     * const { data, error } = await supabase
     *   .storage
     *   .vectors
     *   .deleteBucket('embeddings-old')
     * ```
     */
    deleteBucket(vectorBucketName) {
        const _super = Object.create(null, {
            deleteBucket: { get: () => super.deleteBucket }
        });
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            return _super.deleteBucket.call(this, vectorBucketName);
        });
    }
}
exports.StorageVectorsClient = StorageVectorsClient;
/**
 *
 * @alpha
 *
 * Scoped client for operations within a specific vector bucket
 * Provides index management and access to vector operations
 *
 * **Public alpha:** This API is part of a public alpha release and may not be available to your account type.
 */
class VectorBucketScope extends VectorIndexApi_1.default {
    /**
     * @alpha
     *
     * Creates a helper that automatically scopes all index operations to the provided bucket.
     *
     * **Public alpha:** This API is part of a public alpha release and may not be available to your account type.
     *
     * @category Vector Buckets
     * @example
     * ```typescript
     * const bucket = supabase.storage.vectors.from('embeddings-prod')
     * ```
     */
    constructor(url, headers, vectorBucketName, fetch) {
        super(url, headers, fetch);
        this.vectorBucketName = vectorBucketName;
    }
    /**
     *
     * @alpha
     *
     * Creates a new vector index in this bucket
     * Convenience method that automatically includes the bucket name
     *
     * **Public alpha:** This API is part of a public alpha release and may not be available to your account type.
     *
     * @category Vector Buckets
     * @param options - Index configuration (vectorBucketName is automatically set)
     * @returns Promise with empty response on success or error
     *
     * @example
     * ```typescript
     * const bucket = supabase.storage.vectors.from('embeddings-prod')
     * await bucket.createIndex({
     *   indexName: 'documents-openai',
     *   dataType: 'float32',
     *   dimension: 1536,
     *   distanceMetric: 'cosine',
     *   metadataConfiguration: {
     *     nonFilterableMetadataKeys: ['raw_text']
     *   }
     * })
     * ```
     */
    createIndex(options) {
        const _super = Object.create(null, {
            createIndex: { get: () => super.createIndex }
        });
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            return _super.createIndex.call(this, Object.assign(Object.assign({}, options), { vectorBucketName: this.vectorBucketName }));
        });
    }
    /**
     *
     * @alpha
     *
     * Lists indexes in this bucket
     * Convenience method that automatically includes the bucket name
     *
     * **Public alpha:** This API is part of a public alpha release and may not be available to your account type.
     *
     * @category Vector Buckets
     * @param options - Listing options (vectorBucketName is automatically set)
     * @returns Promise with response containing indexes array and pagination token or error
     *
     * @example
     * ```typescript
     * const bucket = supabase.storage.vectors.from('embeddings-prod')
     * const { data } = await bucket.listIndexes({ prefix: 'documents-' })
     * ```
     */
    listIndexes() {
        const _super = Object.create(null, {
            listIndexes: { get: () => super.listIndexes }
        });
        return tslib_1.__awaiter(this, arguments, void 0, function* (options = {}) {
            return _super.listIndexes.call(this, Object.assign(Object.assign({}, options), { vectorBucketName: this.vectorBucketName }));
        });
    }
    /**
     *
     * @alpha
     *
     * Retrieves metadata for a specific index in this bucket
     * Convenience method that automatically includes the bucket name
     *
     * **Public alpha:** This API is part of a public alpha release and may not be available to your account type.
     *
     * @category Vector Buckets
     * @param indexName - Name of the index to retrieve
     * @returns Promise with index metadata or error
     *
     * @example
     * ```typescript
     * const bucket = supabase.storage.vectors.from('embeddings-prod')
     * const { data } = await bucket.getIndex('documents-openai')
     * console.log('Dimension:', data?.index.dimension)
     * ```
     */
    getIndex(indexName) {
        const _super = Object.create(null, {
            getIndex: { get: () => super.getIndex }
        });
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            return _super.getIndex.call(this, this.vectorBucketName, indexName);
        });
    }
    /**
     *
     * @alpha
     *
     * Deletes an index from this bucket
     * Convenience method that automatically includes the bucket name
     *
     * **Public alpha:** This API is part of a public alpha release and may not be available to your account type.
     *
     * @category Vector Buckets
     * @param indexName - Name of the index to delete
     * @returns Promise with empty response on success or error
     *
     * @example
     * ```typescript
     * const bucket = supabase.storage.vectors.from('embeddings-prod')
     * await bucket.deleteIndex('old-index')
     * ```
     */
    deleteIndex(indexName) {
        const _super = Object.create(null, {
            deleteIndex: { get: () => super.deleteIndex }
        });
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            return _super.deleteIndex.call(this, this.vectorBucketName, indexName);
        });
    }
    /**
     *
     * @alpha
     *
     * Access operations for a specific index within this bucket
     * Returns a scoped client for vector data operations
     *
     * **Public alpha:** This API is part of a public alpha release and may not be available to your account type.
     *
     * @category Vector Buckets
     * @param indexName - Name of the index
     * @returns Index-scoped client with vector data operations
     *
     * @example
     * ```typescript
     * const index = supabase.storage.vectors.from('embeddings-prod').index('documents-openai')
     *
     * // Insert vectors
     * await index.putVectors({
     *   vectors: [
     *     { key: 'doc-1', data: { float32: [...] }, metadata: { title: 'Intro' } }
     *   ]
     * })
     *
     * // Query similar vectors
     * const { data } = await index.queryVectors({
     *   queryVector: { float32: [...] },
     *   topK: 5
     * })
     * ```
     */
    index(indexName) {
        return new VectorIndexScope(this.url, this.headers, this.vectorBucketName, indexName, this.fetch);
    }
}
exports.VectorBucketScope = VectorBucketScope;
/**
 *
 * @alpha
 *
 * Scoped client for operations within a specific vector index
 * Provides vector data operations (put, get, list, query, delete)
 *
 * **Public alpha:** This API is part of a public alpha release and may not be available to your account type.
 */
class VectorIndexScope extends VectorDataApi_1.default {
    /**
     *
     * @alpha
     *
     * Creates a helper that automatically scopes all vector operations to the provided bucket/index names.
     *
     * **Public alpha:** This API is part of a public alpha release and may not be available to your account type.
     *
     * @category Vector Buckets
     * @example
     * ```typescript
     * const index = supabase.storage.vectors.from('embeddings-prod').index('documents-openai')
     * ```
     */
    constructor(url, headers, vectorBucketName, indexName, fetch) {
        super(url, headers, fetch);
        this.vectorBucketName = vectorBucketName;
        this.indexName = indexName;
    }
    /**
     *
     * @alpha
     *
     * Inserts or updates vectors in this index
     * Convenience method that automatically includes bucket and index names
     *
     * **Public alpha:** This API is part of a public alpha release and may not be available to your account type.
     *
     * @category Vector Buckets
     * @param options - Vector insertion options (bucket and index names automatically set)
     * @returns Promise with empty response on success or error
     *
     * @example
     * ```typescript
     * const index = supabase.storage.vectors.from('embeddings-prod').index('documents-openai')
     * await index.putVectors({
     *   vectors: [
     *     {
     *       key: 'doc-1',
     *       data: { float32: [0.1, 0.2, ...] },
     *       metadata: { title: 'Introduction', page: 1 }
     *     }
     *   ]
     * })
     * ```
     */
    putVectors(options) {
        const _super = Object.create(null, {
            putVectors: { get: () => super.putVectors }
        });
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            return _super.putVectors.call(this, Object.assign(Object.assign({}, options), { vectorBucketName: this.vectorBucketName, indexName: this.indexName }));
        });
    }
    /**
     *
     * @alpha
     *
     * Retrieves vectors by keys from this index
     * Convenience method that automatically includes bucket and index names
     *
     * **Public alpha:** This API is part of a public alpha release and may not be available to your account type.
     *
     * @category Vector Buckets
     * @param options - Vector retrieval options (bucket and index names automatically set)
     * @returns Promise with response containing vectors array or error
     *
     * @example
     * ```typescript
     * const index = supabase.storage.vectors.from('embeddings-prod').index('documents-openai')
     * const { data } = await index.getVectors({
     *   keys: ['doc-1', 'doc-2'],
     *   returnMetadata: true
     * })
     * ```
     */
    getVectors(options) {
        const _super = Object.create(null, {
            getVectors: { get: () => super.getVectors }
        });
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            return _super.getVectors.call(this, Object.assign(Object.assign({}, options), { vectorBucketName: this.vectorBucketName, indexName: this.indexName }));
        });
    }
    /**
     *
     * @alpha
     *
     * Lists vectors in this index with pagination
     * Convenience method that automatically includes bucket and index names
     *
     * **Public alpha:** This API is part of a public alpha release and may not be available to your account type.
     *
     * @category Vector Buckets
     * @param options - Listing options (bucket and index names automatically set)
     * @returns Promise with response containing vectors array and pagination token or error
     *
     * @example
     * ```typescript
     * const index = supabase.storage.vectors.from('embeddings-prod').index('documents-openai')
     * const { data } = await index.listVectors({
     *   maxResults: 500,
     *   returnMetadata: true
     * })
     * ```
     */
    listVectors() {
        const _super = Object.create(null, {
            listVectors: { get: () => super.listVectors }
        });
        return tslib_1.__awaiter(this, arguments, void 0, function* (options = {}) {
            return _super.listVectors.call(this, Object.assign(Object.assign({}, options), { vectorBucketName: this.vectorBucketName, indexName: this.indexName }));
        });
    }
    /**
     *
     * @alpha
     *
     * Queries for similar vectors in this index
     * Convenience method that automatically includes bucket and index names
     *
     * **Public alpha:** This API is part of a public alpha release and may not be available to your account type.
     *
     * @category Vector Buckets
     * @param options - Query options (bucket and index names automatically set)
     * @returns Promise with response containing matches array of similar vectors ordered by distance or error
     *
     * @example
     * ```typescript
     * const index = supabase.storage.vectors.from('embeddings-prod').index('documents-openai')
     * const { data } = await index.queryVectors({
     *   queryVector: { float32: [0.1, 0.2, ...] },
     *   topK: 5,
     *   filter: { category: 'technical' },
     *   returnDistance: true,
     *   returnMetadata: true
     * })
     * ```
     */
    queryVectors(options) {
        const _super = Object.create(null, {
            queryVectors: { get: () => super.queryVectors }
        });
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            return _super.queryVectors.call(this, Object.assign(Object.assign({}, options), { vectorBucketName: this.vectorBucketName, indexName: this.indexName }));
        });
    }
    /**
     *
     * @alpha
     *
     * Deletes vectors by keys from this index
     * Convenience method that automatically includes bucket and index names
     *
     * **Public alpha:** This API is part of a public alpha release and may not be available to your account type.
     *
     * @category Vector Buckets
     * @param options - Deletion options (bucket and index names automatically set)
     * @returns Promise with empty response on success or error
     *
     * @example
     * ```typescript
     * const index = supabase.storage.vectors.from('embeddings-prod').index('documents-openai')
     * await index.deleteVectors({
     *   keys: ['doc-1', 'doc-2', 'doc-3']
     * })
     * ```
     */
    deleteVectors(options) {
        const _super = Object.create(null, {
            deleteVectors: { get: () => super.deleteVectors }
        });
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            return _super.deleteVectors.call(this, Object.assign(Object.assign({}, options), { vectorBucketName: this.vectorBucketName, indexName: this.indexName }));
        });
    }
}
exports.VectorIndexScope = VectorIndexScope;
//# sourceMappingURL=StorageVectorsClient.js.map